/************************************************************************
 * @file: AudioTypes.h
 *
 * @version: 0.1
 *
 * @description: This header file contains all the typedef, enum and struct
 * required by RTA component. This is a common header file used by all
 * RTA component.
 * Also contains helper macros for converting time to frames, frame to time.
 *
 * @authors: Jens Lorenz, jlorenz@de.adit-jv.com 2015
 *           Thouseef Ahamed, tahamed@de.adit-jv.com 2105
 *           Vijay Palaniswamy, vijay.palaniswamy@in.bosch.com 2015
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef _ADIT_UTILITY_AUDIOTYPES_H_
#define _ADIT_UTILITY_AUDIOTYPES_H_

#include <ostream>

namespace adit
{

namespace utility
{

namespace audio
{

enum class AudioError
{
    OK = 0,         /**< Operation is success. */
    NOMEM,          /**< Out of memory. */
    UNSUPPORTED,    /**< The given data or file type is not supported. */
    BUSY,           /**< Operation already in progress */
    FAILURE,        /**< Operation is failed.  */
    INVALID         /**< Operation is invalid.  */
};

inline std::ostream& operator<<(std::ostream& os, const AudioError error)
{
    const char* string[] = { "OK", "NOMEM", "UNSUPPORTED", "BUSY", "FAILURE", "INVALID" };
    return os << string[static_cast<int>(error)];
}

enum class AudioState
{
    CONTINUE = 0,   /**< Normal operation user data is available. */
    STOP,           /**< Stop streaming.  */
    ABORT           /**< Abort streaming. */
                      /* Note: In playback case, valid user data has to
                       * be provided in order to avoid "click/pop" noise. */
};

inline std::ostream& operator<<(std::ostream& os, const AudioState state)
{
    const char* string[] = { "CONTINUE", "STOP", "ABORT" };
    return os << string[static_cast<int>(state)];
}

/* AudioFormat enum is derived from ALSA specification */
enum AudioFormat
{
    UNKNOWN = -1,
    S8 = 0,         /**< Signed 8 bit No endian.        */
    U8,             /**< Unsigned 8 bit No endian.      */
    S16_LE,         /**< Signed 16 bit Little endian.   */
    S16_BE,         /**< Signed 16 bit Big endian.      */
    U16_LE,         /**< Unsigned 16 bit Little endian. */
    U16_BE,         /**< Unsigned 16 bit Big endian.    */
    S24_LE,         /**< Signed 24 bit Little endian.   */
    S24_BE,         /**< Signed 24 bit Big endian.      */
    U24_LE,         /**< Unsigned 24 bit Little endian. */
    U24_BE,         /**< Unsigned 24 bit Big endian.    */
    S32_LE,         /**< Signed 32 bit Little endian.   */
    S32_BE,         /**< Signed 32 bit Big endian.      */
    U32_LE,         /**< Unsigned 32 bit Little endian. */
    U32_BE,         /**< Unsigned 32 bit Big endian.    */
    FLOAT32_LE,     /**< Float 32 bit Little endian.    */
    FLOAT32_BE,     /**< Float 32 bit Big endian.       */
    FLOAT64_LE,     /**< Float 64 bit Little endian.    */
    FLOAT64_BE      /**< Float 64 bit Big endian.       */
};

inline std::ostream& operator<<(std::ostream& os, AudioFormat format)
{
    const char* string[] = { "UNKNOWN", "S8", "U8", "S16_LE", "S16_BE", "U16_LE", "U16_BE",
        "S24_LE", "S24_BE", "U24_LE", "U24_BE", "S32_LE", "S32_BE", "U32_LE", "U32_BE",
        "FLOAT32_LE", "FLOAT32_BE", "FLOAT64_LE", "FLOAT64_BE" };
    return os << string[static_cast<int>(format)+1];
}

inline uint32_t getBytesPerSample(AudioFormat format)
{
    switch (format) {
        case S8:
        case U8:
            return 1;
        case S16_LE:
        case S16_BE:
        case U16_LE:
        case U16_BE:
            return 2;
        case S24_LE:
        case S24_BE:
        case U24_LE:
        case U24_BE:
            return 3;
        case S32_LE:
        case S32_BE:
        case U32_LE:
        case U32_BE:
        case FLOAT32_LE:
        case FLOAT32_BE:
            return 4;
        case FLOAT64_LE:
        case FLOAT64_BE:
            return 8;
        default:
            return 0;
    }
}

enum class StreamDirection
{
    UNKNOWN = -1,   /**< Stream direction unknown */
    IN = 0,         /**< Stream direction In, capture */
    OUT,            /**< Stream direction Out, playback */
};

inline std::ostream& operator<<(std::ostream& os, StreamDirection dir)
{
    const char* string[] = { "UNKNOWN", "IN", "OUT" };
    return os << string[static_cast<int>(dir)+1];
}

enum class FadeMode
{
    UNKNOWN = -1,   /**< Fade Mode Unknown */
    IN = 0,         /**< Fade-In */
    OUT,            /**< Fade-Out */
};

inline std::ostream& operator<<(std::ostream& os, FadeMode mode)
{
    const char* string[] = { "UNKNOWN", "IN", "OUT" };
    return os << string[static_cast<int>(mode)+1];
}

} /* namespace audio */

} /* namespace utility */

} /* namespace adit */

#endif /* _ADIT_UTILITY_AUDIOTYPES_H_ */
